/* -*- mode: C++; tab-width: 4 -*- */
/* ===================================================================== *\
	Copyright (c) 2001 Palm, Inc. or its subsidiaries.
	All rights reserved.

	This file is part of the Palm OS Emulator.

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
\* ===================================================================== */

#include "EmCommon.h"
#include "EmSession.h"
#include "EmRegsEZVisor.h"
#include "EmRegsVZMorf.h"
#include "EmRegsS1D13A03.h"
#include "EmRegsVZPrv.h"
#include "EmBankROM.h"			// EmBankROM::GetMemoryStart
#include "EmBankRegs.h"
#include "EmHAL.h"
#include "Logging.h"			// LogAppendMsg
#define PRINTF	if (0) ; else LogAppendMsg


/************************************************************************
 * Port B Bit settings
 ************************************************************************/
#define hwrMimoPortBUnused0					0x01	// ( ) Unused GPIO (driven low)
#define hwrMimoPortBUnused5					0x20	// ( ) Unused GPIO (driven low)
#define hwrMimoPortBUnused6					0x40	// ( ) Unused GPIO (driven low)

/************************************************************************
 * Port C Bit settings
 ************************************************************************/
#define hwrMimoPortCKbdCol0		        	0x01    // ( ) Keyboard Row 0
#define hwrMimoPortCKbdCol1     		   	0x02    // ( ) Keyboard Row 1
#define hwrMimoPortCKbdCol2        			0x04    // ( ) Keyboard Row 2
#define hwrMimoPortCKbdCol3       			0x08    // ( ) Keyboard Row 3
#define hwrMimoPortCKbdCol4					0x10    // ( ) Keyboard Row 4
#define hwrMimoPortCKbdCol5					0x20    // ( ) Keyboard Row 5
#define hwrMimoPortCKbdCol6					0x40    // ( ) Keyboard Row 6
#define hwrMimoPortCKbdCol7					0x80    // ( ) Keyboard Row 7

#define hwrMimoPortCKeyBits					0xFF
 

/************************************************************************
 * Port D Bit settings
 ************************************************************************/
#define hwrMimoPortDDock2Off				0x01	// (L) Dock2 input (int0)
#define hwrMimoPortDSDWriteProtectOn		0x02	// (H) SD Write Protect (int1)
#define hwrMimoPortDPowerSwitchOff			0x04	// (L) Power switch (int2)
#define hwrMimoPortDDock1Off				0x08 	// (L) Dock1 input (int3)

#define hwrMimoPortDKbdIrqOff				0x10	// (L) Keyboard Matrix (IRQ1)
#define hwrMimoPortDExtPowerOff    			0x20    // (L) External power indicator (IRQ2)
#define hwrMimoPortDUsbIrqOff				0x40	// (L) USB Interrupt (IRQ3)
#define hwrMimoPortDSDIrqOn					0x80    // 
// XXXPortDKeyBits is the bits in the INT section of port D that are get an
// interrupt when a hardware button is pressed.  This can be power, hotsync, hard
// keys, scroll keys, or qwerty keys.
#define	hwrMimoPortDKeyBits					0x0C 	// (H) All Keyboard Interrupts

// I think this should be all INT bits that can wake up the device?
#define hwrZ328PortDIntMaskMimo				(hwrZ328PortDInt2 | hwrZ328PortDInt3)


/************************************************************************
 * Port E Bit settings
 ************************************************************************/
#define hwrMimoPortELedOn					0x08	// (H) Turn On LED		
#define hwrMimoPortEKbdRow4					0x40    // ( ) Keyboard Row 4
#define hwrMimoPortEKbdRow5					0x80    // ( ) Keyboard Row 5
#define hwrMimoPortEKeyBits					0xC0												//	   ...turn on ASAP!


/************************************************************************
 * Port F Bit settings
 ************************************************************************/
#define hwrMimoPortFLCDEnableOn				0x20	// (H) Keyboard light
#define hwrMimoPortFUSBClockEnableOn		0x40	// (H) USB clock enable. 
													// Epson 13A3 chip's USB subsection 
													// only active when USB clock is 'on'.
/************************************************************************
 * Port G Bit settings
 ************************************************************************/
#define hwrMimoUsbSuspendOff				0x02	// (X) USB Suspend bit - Output low for active
#define	hwrMimoPortGPowerFailIrqOff			0x04	// (L) Power Fail IRQ
#define hwrMimoPortGAdcCsOff				0x20	// (L) A/D Select


/************************************************************************
 * Port J Bit settings
 ************************************************************************/
#define hwrMimoPortJSDPowerOff		0x08

 /************************************************************************
 * Port K Bit settings
 ************************************************************************/
#define hwrMimoPortKKbdRow0        	0x01    // ( ) Keyboard Row 0
#define hwrMimoPortKKbdRow1        	0x02    // ( ) Keyboard Row 1
#define hwrMimoPortKKbdRow2        	0x04    // ( ) Keyboard Row 2
#define hwrMimoPortKKbdRow3       	0x08    // ( ) Keyboard Row 3
#define	hwrMimoPortKKeyBits			0x0F 	// ( ) All Port K Keyboard Rows
#define	hwrMimoPortKKeyModBits		0x02	// ( ) All Port K Keyboard Rows
											//	   with modifier keys on them
#define hwrMimoPortKLCDAdjOn	        0x10	// (H) LCD Contrast Voltage
#define hwrMimoPortKKbdLightOn          0x20
#define	hwrMimoPortKAToDAndLCDVccOff	0x40	// (L) LCD Vcc
#define	hwrMimoPortKLcdResetOff		    0x80	// (L) LCD reset


/************************************************************************
 * Port M Bit settings
 ************************************************************************/
#define hwrMimoPortMIREnableOff				0x20	// (L) Shutdown IR

/************************************************************************
 * Memory mapped I/O
 ************************************************************************/
#define hwrZ328PortDInt0				0x01	
#define hwrZ328PortDInt2				0x04		
#define hwrZ328PortDInt3				0x08	
#define hwrZ328PortDIntMaskMimo			(hwrZ328PortDInt2 | hwrZ328PortDInt3)

// ---------------------------------------------------------------------------
//		Keyboard
// ---------------------------------------------------------------------------

#define numRowsEnUS		6
#define numColsEnUS		8

static const SkinElementType matrixEnUS[numRowsEnUS][numColsEnUS] =
{
	// Row 0
	{
		kElement_None,
		kElement_App1Button,
		kElement_App2Button,
		kElement_App3Button,
		kElement_App4Button,
		kElement_JogSelectButton,
		kElement_JogDownButton,
		kElement_JogUpButton
	},
	// Row 1
	{
		kElement_OptionKey,
		kElement_ShiftKey,
		kElement_PeriodKey,
		kElement_EnterKey,
		kElement_SpaceKey,
		kElement_MenuKey,
		kElement_DownButton,
		kElement_UpButton
	},
	// Row 2
	{
		kElement_OKey,
		kElement_PKey,
		kElement_LKey,
		kElement_BackSpaceKey,
		kElement_None,
		kElement_None,
		kElement_SymbolKey,
		kElement_None
	},
	// Row 3
	{
		kElement_None,
		kElement_ZKey,
		kElement_XKey,
		kElement_CKey,
		kElement_VKey,
		kElement_BKey,
		kElement_NKey,
		kElement_MKey
	},
	// Row 4
	{
		kElement_AKey,
		kElement_SKey,
		kElement_DKey,
		kElement_FKey,
		kElement_GKey,
		kElement_HKey,
		kElement_JKey,
		kElement_KKey
	},
	// Row 5
	{
		kElement_QKey,
		kElement_WKey,
		kElement_EKey,
		kElement_RKey,
		kElement_TKey,
		kElement_YKey,
		kElement_UKey,
		kElement_IKey
	}
};


// ---------------------------------------------------------------------------
//		 EmRegsVZMorf::EmRegsVZMorf
// ---------------------------------------------------------------------------

EmRegsVZMorf::EmRegsVZMorf (void)
{
	EmRegsVZVisorPlatinum::EmRegsVZVisorPlatinum ();
	keyMatrixP = new EmRegsVZMorfKeyMatrix(false);
}

// ---------------------------------------------------------------------------
//		 EmRegsVZMorf::~EmRegsVZMorf
// ---------------------------------------------------------------------------

EmRegsVZMorf::~EmRegsVZMorf (void)
{
	delete keyMatrixP;
}


// ---------------------------------------------------------------------------
//		 EmRegsVZMorf::GetLCDScreenOn
// ---------------------------------------------------------------------------

Bool EmRegsVZMorf::GetLCDScreenOn (void)
{
	return EmHALHandler::GetLCDScreenOn ();
	//return (READ_REGISTER (portKData) & hwrMorfPortCLCDEnableOn) != 0;
}


// ---------------------------------------------------------------------------
//		 EmRegsVZMorf::GetLCDBacklightOn
// ---------------------------------------------------------------------------

Bool EmRegsVZMorf::GetLCDBacklightOn (void)
{

	return EmHALHandler::GetLCDBacklightOn ();
	//return !(READ_REGISTER (portFData) & hwrMorfPortCKbdLightOn) == 0;
}

// ---------------------------------------------------------------------------
//		 EmRegsVZGenericColor::GetLCDHasFrame
// ---------------------------------------------------------------------------

Bool EmRegsVZMorf::GetLCDHasFrame (void)
{
	// Override the Dragonball version and let the SED 1376 handle it.

	return EmHALHandler::GetLCDHasFrame ();
}


// ---------------------------------------------------------------------------
//		 EmRegsVZGenericColor::GetLCDBeginEnd
// ---------------------------------------------------------------------------

void EmRegsVZMorf::GetLCDBeginEnd (emuptr& begin, emuptr& end)
{
	// Override the Dragonball version and let the SED 1376 handle it.

	EmHALHandler::GetLCDBeginEnd (begin, end);
}


// ---------------------------------------------------------------------------
//		 EmRegsVZGenericColor::GetLCDScanlines
// ---------------------------------------------------------------------------

void EmRegsVZMorf::GetLCDScanlines (EmScreenUpdateInfo& info)
{
	// Override the Dragonball version and let the SED 1376 handle it.

	EmHALHandler::GetLCDScanlines (info);
}

// ---------------------------------------------------------------------------
//		 EmRegsVZMorf::GetLEDState
// ---------------------------------------------------------------------------

UInt16 EmRegsVZMorf::GetLEDState (void)
{
	UInt16	result		= kLEDOff;

	if (READ_REGISTER (portEData) & hwrMimoPortELedOn != 0)
	{
		result |= hwrMimoPortELedOn;
	} 	

	return result;
}


// ---------------------------------------------------------------------------
//		 EmRegsVZMorf::GetPortInputValue
//
//	Returns the GPIO value of the given port.
// ---------------------------------------------------------------------------

UInt8 EmRegsVZMorf::GetPortInputValue (int port)
{
	UInt8	result = EmRegsVZ::GetPortInputValue (port);
	
	if (port == 'C')
	{
		UInt8	portCData		= READ_REGISTER (portCData);
		UInt8	portCDir		= READ_REGISTER (portCDir);

		// We return a 1 for any key bit that is an output and
		// is driven low. This is a bit weird, but we don't
		// have access to the direction registers from the
		// matrix driver, so we hack this a bit.

		UInt8	portEData		= READ_REGISTER (portEData);
		UInt8	portEDir		= READ_REGISTER (portEDir);
		UInt8	portEMask		= ~portEData & portEDir & hwrMimoPortEKeyBits;
		
		UInt8	portKData		= READ_REGISTER (portKData);
		UInt8	portKDir		= READ_REGISTER (portKDir);
		UInt8	portKMask		= ~portKData & portKDir & hwrMimoPortKKeyBits;
		
		UInt32 curMatrix = keyMatrixP->ReadMatrix(portKMask, portEMask);

		result |= hwrMimoPortCKeyBits & (~curMatrix);
	}
	else if (port == 'D')
	{ 
		result &= ~hwrMimoPortDDock2Off;
		result |= hwrMimoPortDSDWriteProtectOn;
		result |= hwrMimoPortDSDIrqOn;
		result |= hwrMimoPortDExtPowerOff;
		//result &= ~hwrMimoPortDUsbIrqOff;
	}
	else if (port == 'E')
	{
		// We return a 1 for any key bit that is an output and
		// is driven low. This is a bit weird, but we don't
		// have access to the direction registers from the
		// matrix driver, so we hack this a bit.
		
		/*
			Fake out the port E model ID scheme.
 
			See comments below. We need to provide an ID of zero here to cause
			the device software to look at port J instead.
		*/

		uint8	modelOutPin		= 0x04;		// EMUIRQ (IRQ7)
		uint8	modelInPins		= 0x07;     // SPI pins on Visor

		uint8	portGData		= READ_REGISTER (portGData);
		uint8	portGDir		= READ_REGISTER (portGDir);
		uint8	portGSelect		= READ_REGISTER (portGSelect);

		if (((portGData & modelOutPin) == 0) &&
			((portGDir & modelOutPin) == modelOutPin) &&
			((portGSelect & modelOutPin) == modelOutPin))
		{
			result &= ~modelInPins;
			result |= ~(0x00) & modelInPins; // Expects inverted output.
		}
	}
	else if (port == 'G')
	{
		// Make sure that hwrMorfPortGPowerFailIrqOff is set.  If it's clear,
		// the battery code will make the device go to sleep immediately.

		result |= hwrMimoPortGPowerFailIrqOff;
	}

	else if (port == 'J')
	{
		/*
			Support the port J model ID scheme.

			If we are on a VZ device with a model ID (low three bits) of zero,
			then the device will read port J to get the lowest two model ID
			bits. The highest bit is currently always zero. We always return
			the proper bits here (port J is not used for any other purpose).
		*/

		uint8	modelInPins		= 0x0C;

		result &= ~modelInPins;
		result |= gSession->GetDevice ().HardwareID () & modelInPins;
		//result |= hwrMimoPortJSDPowerOff;
	}

	return result;
}


// ---------------------------------------------------------------------------
//		 EmRegsVZMorf::GetPortInternalValue
//
//	Returns the "dedicated" pin value of the given port.
// ---------------------------------------------------------------------------

UInt8 EmRegsVZMorf::GetPortInternalValue (int port)
{
	UInt8	result = EmRegsVZ::GetPortInternalValue (port);

	if (port == 'D')
	{
		// Ensure that bit hwrMorfPortDDock1IrqOff is set.
		// If it's clear, HotSync will sync via the modem instead
		// of the serial port.

		result &= ~hwrMimoPortDDock1Off;
		//result &= ~hwrMimoPortDPowerSwitchOff;
		//result &= ~hwrMimoPortDUsbIrqOff;
		result |= hwrMimoPortDKbdIrqOff;
		result |= hwrMimoPortDExtPowerOff;
		result |= hwrMimoPortDSDIrqOn;
		result |= hwrMimoPortDSDWriteProtectOn;
	}
	else if (port == 'G')
	{
		// Make sure that hwrMorfPortGPowerFailIrqOff is set.  If it's clear,
		// the battery code will make the device go to sleep immediately.

		result |= hwrMimoPortGPowerFailIrqOff;
	}

	return result;
}


// ---------------------------------------------------------------------------
//		 EmRegsVZMorf::ButtonEvent
// ---------------------------------------------------------------------------

void EmRegsVZMorf::ButtonEvent (SkinElementType button, Bool buttonIsDown)
{
	EmAssert (keyMatrixP);

	// This one doesn't go through the key matrix.

	if (button == kElement_LidSwitch)
	{
		// DOLATER: Add lid toggling support here.

		return;
	}

	if (buttonIsDown)
	{
		// Set the matrix state.

		Bool interruptNeeded = keyMatrixP->SetMatrixState (true /*state*/, button);

		if (interruptNeeded)
		{
			// Set up a key interrupt which will initiate a scan.

			UInt16	intPendingHi	= READ_REGISTER (intPendingHi);
			intPendingHi |= hwrVZ328IntHiIRQ1;
			WRITE_REGISTER (intPendingHi, intPendingHi);

			EmRegsVZ::UpdateInterrupts ();
		}
	}
	else
	{
		// Clear the matrix state.

		keyMatrixP->SetMatrixState (false /*state*/, button);
	}
}


// ---------------------------------------------------------------------------
//		 EmRegsVZMorf::GetKeyBits
// ---------------------------------------------------------------------------

UInt8 EmRegsVZMorf::GetKeyBits (void)
{
	return 0;
}


// ---------------------------------------------------------------------------
//		 EmRegsVZMorfKeyMatrix::EmRegsVZMorfKeyMatrix
// ---------------------------------------------------------------------------

EmRegsVZMorfKeyMatrix::EmRegsVZMorfKeyMatrix (bool SheaID)
{
	this->numRows = numRowsEnUS;
	this->numCols = numColsEnUS;
	this->ResetMatrixState ();
}


// ---------------------------------------------------------------------------
//		 EmRegsVZMorfKeyMatrix::~EmRegsVZMorfKeyMatrix
// ---------------------------------------------------------------------------

EmRegsVZMorfKeyMatrix::~EmRegsVZMorfKeyMatrix (void)
{
}


// ---------------------------------------------------------------------------
//		 EmRegsVZMorfKeyMatrix::ReadMatrix
// ---------------------------------------------------------------------------

#define READ_ROW(n)	  do {											  \
							result |= this->GetColumnBits (n);		  \
							numRead++;								  \
						 } while (0);

UInt32 EmRegsVZMorfKeyMatrix::ReadMatrix (UInt8 portKMask, UInt8 portEMask)
{
	//UInt8	portKData = EmHAL::GetPortInternalValue ('K');
	//UInt8	portEData = EmHAL::GetPortInternalValue ('E');

	UInt16 result = 0x0000;
	UInt16 numRead = 0;

	if (portKMask & hwrMimoPortKKbdRow0) READ_ROW (0);
	if (portKMask & hwrMimoPortKKbdRow1) READ_ROW (1);
	if (portKMask & hwrMimoPortKKbdRow2) READ_ROW (2);
	if (portKMask & hwrMimoPortKKbdRow3) READ_ROW (3);
	if (portEMask & hwrMimoPortEKbdRow4) READ_ROW (4);
	if (portEMask & hwrMimoPortEKbdRow5) READ_ROW (5);

	return result;
}


// ---------------------------------------------------------------------------
//		 EmRegsVZMorfKeyMatrix::ReadMatrix
// ---------------------------------------------------------------------------
/*
UInt32 EmRegsVZMorfKeyMatrix::ReadMatrix ()
{
	
	UInt32 result = 0x00000000;

	for (int row = 0; row < this->numRows; row++)
	{
		for (int col = 0 ; col < this->numCols; col++)
		{
			if (matrixState[row][col] == 1)
			{
			   result |= (1 << col);
			   result |= ((1 << row) << 16);
			}
		}
	}
	
	return result;
}
*/

// ---------------------------------------------------------------------------
//		 EmRegsVZMorfKeyMatrix::GetColumnBits
// ---------------------------------------------------------------------------

UInt8 EmRegsVZMorfKeyMatrix::GetColumnBits (int row)
{
	if (row >= this->numRows)
	{
		EmAssert (0);
		return 0;
	}

	UInt16 result = 0;

	for (int col = 0 ; col < this->numCols; col++)
	{
		if (this->matrixState[row][col] == 1)
		{
			result |= 1 << col;
		}
	}

	return result;
}


// ---------------------------------------------------------------------------
//		 EmRegsVZMorfKeyMatrix::ResetMatrixState
// ---------------------------------------------------------------------------

void EmRegsVZMorfKeyMatrix::ResetMatrixState ()
{
	for (int row = 0; row < this->numRows; row++)
		for (int col = 0; col < this->numCols; col++)
			this->matrixState[row][col] = 0;
}

// ---------------------------------------------------------------------------
//		 EmRegsVZMorfKeyMatrix::SetMatrixState
// ---------------------------------------------------------------------------

Bool EmRegsVZMorfKeyMatrix::SetMatrixState (int state, SkinElementType element)
{
	Bool result = false;

	for (int row = 0; row < this->numRows; row++)
	{
		for (int col = 0 ; col < this->numCols; col++)
		{
			if (matrixEnUS[row][col] == element)
			{
				// If this is the only thing "down" in the matrix on this row, then
				// we return true to signal that an interrupt is needed.

				if (this->GetColumnBits (row) == 0)
				{
					result = true;
				}

				this->matrixState[row][col] = state;
			}
		}
	}

	return result;
}

